<?php
require_once 'config.php';

if(!isLoggedIn()) {
    header("Location: login.php");
    exit();
}

if(!isActiveUser()) {
    header("Location: activate.php");
    exit();
}

$user_id = $_SESSION['user_id'];

// Check if checkout data exists
if(!isset($_SESSION['checkout_data']) || empty($_SESSION['checkout_data']['items'])) {
    header("Location: cart.php");
    exit();
}

$checkout_data = $_SESSION['checkout_data'];
$cart_items = $checkout_data['items'];
$total_amount = $checkout_data['total'];

// Process checkout
if($_SERVER['REQUEST_METHOD'] == 'POST') {
    $telegram = sanitize($_POST['telegram_username']);
    $whatsapp = sanitize($_POST['whatsapp_number']);
    $customer_name = sanitize($_POST['customer_name']);
    $notes = sanitize($_POST['notes']);
    
    if(empty($telegram) && empty($whatsapp)) {
        $error = "Please provide at least Telegram username or WhatsApp number";
    } else {
        $user_balance = getUserBalance($user_id);
        
        if($user_balance >= $total_amount) {
            $conn->begin_transaction();
            
            try {
                // Deduct balance
                $conn->query("UPDATE users SET balance = balance - $total_amount WHERE id = $user_id");
                
                // Create orders for each product
                foreach($cart_items as $item) {
                    $order_stmt = $conn->prepare("
                        INSERT INTO orders (user_id, product_id, amount, telegram_username, whatsapp_number, customer_name, admin_message, status) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, 'pending')
                    ");
                    $order_stmt->bind_param(
                        "iidssss", 
                        $user_id, 
                        $item['id'], 
                        $item['sale_price'], 
                        $telegram, 
                        $whatsapp, 
                        $customer_name,
                        $notes
                    );
                    $order_stmt->execute();
                }
                
                // Record transaction
                $conn->query("
                    INSERT INTO transactions (user_id, type, amount, description, status) 
                    VALUES ($user_id, 'product_purchase', $total_amount, 'Purchase of " . count($cart_items) . " items', 'completed')
                ");
                
                $conn->commit();
                
                // Clear cart and checkout data
                unset($_SESSION['cart']);
                unset($_SESSION['checkout_data']);
                
                $success = "Checkout successful! Your order has been placed.";
                header("refresh:3;url=my-orders.php");
            } catch (Exception $e) {
                $conn->rollback();
                $error = "Checkout failed. Please try again.";
            }
        } else {
            $error = "Insufficient balance. Please deposit first.";
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Checkout - Project Sale</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    <style>
        .checkout-step {
            transition: all 0.3s ease;
        }
        .checkout-step.active {
            transform: scale(1.05);
        }
        .order-item {
            transition: all 0.2s ease;
        }
        .order-item:hover {
            background-color: #f9fafb;
        }
    </style>
</head>
<body class="bg-gray-50">
    <!-- Include Header -->
    <?php include 'includes/header.php'; ?>
    
    <div class="flex min-h-screen">
        <!-- Sidebar -->
        <div class="hidden md:block w-64 bg-white shadow-lg">
            <?php include 'includes/sidebar.php'; ?>
        </div>
        
        <!-- Main Content -->
        <div class="flex-1 p-6">
            <!-- Breadcrumb -->
            <div class="mb-6">
                <nav class="flex" aria-label="Breadcrumb">
                    <ol class="inline-flex items-center space-x-1 md:space-x-3">
                        <li class="inline-flex items-center">
                            <a href="index.php" class="inline-flex items-center text-sm text-gray-700 hover:text-blue-600">
                                <i class="fas fa-home mr-2"></i> Home
                            </a>
                        </li>
                        <li>
                            <div class="flex items-center">
                                <i class="fas fa-chevron-right text-gray-400 text-xs"></i>
                                <a href="cart.php" class="ml-1 text-sm text-gray-700 hover:text-blue-600 md:ml-2">Cart</a>
                            </div>
                        </li>
                        <li aria-current="page">
                            <div class="flex items-center">
                                <i class="fas fa-chevron-right text-gray-400 text-xs"></i>
                                <span class="ml-1 text-sm text-gray-500 md:ml-2 font-medium">Checkout</span>
                            </div>
                        </li>
                    </ol>
                </nav>
            </div>
            
            <!-- Header -->
            <div class="mb-8">
                <h1 class="text-3xl font-bold text-gray-800 mb-2">
                    <i class="fas fa-cash-register mr-2"></i> Checkout
                </h1>
                <p class="text-gray-600">Complete your purchase with secure payment</p>
            </div>
            
            <!-- Messages -->
            <?php if(isset($error)): ?>
                <div class="bg-red-50 border-l-4 border-red-500 p-4 mb-6 rounded-lg animate__animated animate__shakeX">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-3 text-xl"></i>
                        <span class="text-red-700 font-medium"><?php echo $error; ?></span>
                    </div>
                </div>
            <?php endif; ?>
            
            <?php if(isset($success)): ?>
                <div class="bg-green-50 border-l-4 border-green-500 p-4 mb-6 rounded-lg animate__animated animate__fadeIn">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-3 text-xl"></i>
                        <span class="text-green-700 font-medium"><?php echo $success; ?></span>
                    </div>
                </div>
            <?php endif; ?>
            
            <!-- Checkout Steps -->
            <div class="mb-8">
                <div class="flex justify-between items-center relative">
                    <!-- Progress Line -->
                    <div class="absolute top-4 left-0 right-0 h-1 bg-gray-200 z-0"></div>
                    <div class="absolute top-4 left-0 right-0 h-1 bg-green-500 z-10" style="width: 66%"></div>
                    
                    <!-- Steps -->
                    <div class="checkout-step flex flex-col items-center relative z-20">
                        <div class="w-10 h-10 bg-green-500 rounded-full flex items-center justify-center text-white font-bold mb-2">
                            <i class="fas fa-check"></i>
                        </div>
                        <span class="text-sm font-medium text-green-600">Cart</span>
                    </div>
                    
                    <div class="checkout-step active flex flex-col items-center relative z-20">
                        <div class="w-12 h-12 bg-gradient-to-r from-blue-500 to-indigo-600 rounded-full flex items-center justify-center text-white font-bold mb-2 shadow-lg">
                            <i class="fas fa-user"></i>
                        </div>
                        <span class="text-sm font-medium text-blue-600">Details</span>
                    </div>
                    
                    <div class="checkout-step flex flex-col items-center relative z-20">
                        <div class="w-10 h-10 bg-gray-300 rounded-full flex items-center justify-center text-gray-500 font-bold mb-2">
                            <i class="fas fa-check"></i>
                        </div>
                        <span class="text-sm font-medium text-gray-500">Complete</span>
                    </div>
                </div>
            </div>
            
            <!-- Checkout Content -->
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                <!-- Order Details -->
                <div class="lg:col-span-2 space-y-6">
                    <!-- Contact Information -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden animate__animated animate__fadeInLeft">
                        <div class="bg-gradient-to-r from-blue-500 to-indigo-600 px-6 py-4">
                            <h3 class="text-xl font-bold text-white">
                                <i class="fas fa-user-circle mr-2"></i> Contact Information
                            </h3>
                        </div>
                        <div class="p-6">
                            <form method="POST" action="" class="space-y-6">
                                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                    <div>
                                        <label class="block text-gray-700 text-sm font-bold mb-2">
                                            <i class="fas fa-user text-blue-500 mr-2"></i> Full Name
                                        </label>
                                        <input type="text" name="customer_name" required
                                               class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                                               placeholder="Enter your full name"
                                               value="<?php echo $_SESSION['username']; ?>">
                                    </div>
                                    
                                    <div class="md:col-span-2">
                                        <label class="block text-gray-700 text-sm font-bold mb-2">
                                            <i class="fab fa-telegram text-blue-500 mr-2"></i> Telegram Username
                                            <span class="text-gray-500 text-sm font-normal ml-2">(Required for delivery)</span>
                                        </label>
                                        <input type="text" name="telegram_username"
                                               class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                                               placeholder="@username">
                                    </div>
                                    
                                    <div class="md:col-span-2">
                                        <label class="block text-gray-700 text-sm font-bold mb-2">
                                            <i class="fab fa-whatsapp text-green-500 mr-2"></i> WhatsApp Number
                                            <span class="text-gray-500 text-sm font-normal ml-2">(Alternative contact)</span>
                                        </label>
                                        <input type="text" name="whatsapp_number"
                                               class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                                               placeholder="+8801XXXXXXXXX">
                                    </div>
                                    
                                    <div class="md:col-span-2">
                                        <label class="block text-gray-700 text-sm font-bold mb-2">
                                            <i class="fas fa-sticky-note text-purple-500 mr-2"></i> Additional Notes
                                        </label>
                                        <textarea name="notes" rows="3"
                                                  class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                                                  placeholder="Any special instructions or notes for the seller..."></textarea>
                                    </div>
                                </div>
                                
                                <!-- Important Note -->
                                <div class="p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
                                    <div class="flex">
                                        <i class="fas fa-exclamation-triangle text-yellow-500 text-xl mr-3 mt-1"></i>
                                        <div>
                                            <h4 class="font-bold text-yellow-800 mb-2">Important Information</h4>
                                            <ul class="list-disc list-inside text-yellow-700 text-sm space-y-1">
                                                <li>Please provide at least one contact method (Telegram or WhatsApp)</li>
                                                <li>Products will be delivered via provided contact method</li>
                                                <li>Delivery usually takes 5-30 minutes after payment confirmation</li>
                                                <li>Keep your contact information active to receive products</li>
                                            </ul>
                                        </div>
                                    </div>
                                </div>
                        </div>
                    </div>
                    
                    <!-- Order Items -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden animate__animated animate__fadeInLeft" style="animation-delay: 0.1s">
                        <div class="bg-gradient-to-r from-purple-500 to-pink-600 px-6 py-4">
                            <h3 class="text-xl font-bold text-white">
                                <i class="fas fa-boxes mr-2"></i> Order Items (<?php echo count($cart_items); ?>)
                            </h3>
                        </div>
                        <div class="p-6">
                            <div class="space-y-4">
                                <?php foreach($cart_items as $item): ?>
                                <div class="order-item flex items-center p-4 border border-gray-200 rounded-lg hover:border-blue-300 transition duration-300">
                                    <div class="w-16 h-16 rounded-lg overflow-hidden mr-4">
                                        <img src="<?php echo $item['image_url'] ?: 'https://via.placeholder.com/100'; ?>" 
                                             alt="<?php echo htmlspecialchars($item['name']); ?>"
                                             class="w-full h-full object-cover">
                                    </div>
                                    <div class="flex-1">
                                        <h4 class="font-bold text-gray-800">
                                            <?php echo htmlspecialchars($item['name']); ?>
                                        </h4>
                                        <div class="flex items-center justify-between mt-2">
                                            <span class="text-gray-600 text-sm">
                                                Price: ৳<?php echo number_format($item['sale_price'], 2); ?>
                                            </span>
                                            <?php if($item['original_price'] > 0): ?>
                                                <span class="text-gray-400 line-through text-sm">
                                                    ৳<?php echo number_format($item['original_price'], 2); ?>
                                                </span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <div class="text-right">
                                        <p class="text-lg font-bold text-gray-800">
                                            ৳<?php echo number_format($item['sale_price'], 2); ?>
                                        </p>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Order Summary -->
                <div class="space-y-6">
                    <!-- Summary Card -->
                    <div class="bg-white rounded-xl shadow-lg p-6 sticky top-24 animate__animated animate__fadeInRight">
                        <h3 class="text-xl font-bold text-gray-800 mb-6">
                            <i class="fas fa-receipt mr-2"></i> Order Summary
                        </h3>
                        
                        <!-- Summary Details -->
                        <div class="space-y-4 mb-6">
                            <div class="flex justify-between">
                                <span class="text-gray-600">Subtotal</span>
                                <span class="font-medium text-gray-800">
                                    ৳<?php echo number_format($total_amount, 2); ?>
                                </span>
                            </div>
                            
                            <div class="flex justify-between">
                                <span class="text-gray-600">Discount</span>
                                <span class="font-medium text-green-600">৳0.00</span>
                            </div>
                            
                            <div class="flex justify-between">
                                <span class="text-gray-600">Platform Fee</span>
                                <span class="font-medium text-gray-800">৳0.00</span>
                            </div>
                            
                            <div class="border-t pt-4">
                                <div class="flex justify-between items-center">
                                    <span class="text-lg font-bold text-gray-800">Total Amount</span>
                                    <span class="text-3xl font-bold text-blue-600">
                                        ৳<?php echo number_format($total_amount, 2); ?>
                                    </span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- User Balance -->
                        <div class="mb-6 p-4 bg-gray-50 rounded-lg">
                            <div class="flex justify-between items-center mb-2">
                                <span class="text-gray-600">Your Balance:</span>
                                <span class="text-xl font-bold <?php echo getUserBalance($user_id) >= $total_amount ? 'text-green-600' : 'text-red-600'; ?>">
                                    ৳<?php echo number_format(getUserBalance($user_id), 2); ?>
                                </span>
                            </div>
                            
                            <?php if(getUserBalance($user_id) < $total_amount): ?>
                                <div class="mt-3 p-3 bg-red-50 border border-red-200 rounded-lg">
                                    <div class="flex items-center">
                                        <i class="fas fa-exclamation-triangle text-red-500 mr-3"></i>
                                        <div>
                                            <p class="text-red-700 text-sm font-medium">Insufficient Balance</p>
                                            <p class="text-red-600 text-xs">
                                                Need ৳<?php echo number_format($total_amount - getUserBalance($user_id), 2); ?> more
                                            </p>
                                        </div>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                        
                        <!-- Terms and Conditions -->
                        <div class="mb-6">
                            <div class="flex items-start">
                                <input type="checkbox" id="terms" required
                                       class="mt-1 mr-3 h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded">
                                <label for="terms" class="text-sm text-gray-700">
                                    I agree to the 
                                    <a href="#" class="text-blue-600 hover:underline">Terms & Conditions</a> 
                                    and understand that all purchases are final.
                                </label>
                            </div>
                        </div>
                        
                        <!-- Submit Button -->
                        <button type="submit" 
                                class="w-full bg-gradient-to-r from-green-500 to-emerald-600 text-white font-bold py-4 px-6 rounded-lg text-lg hover:from-green-600 hover:to-emerald-700 focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2 transition duration-300 transform hover:scale-105 disabled:opacity-50 disabled:cursor-not-allowed"
                                <?php echo getUserBalance($user_id) < $total_amount ? 'disabled' : ''; ?>>
                            <i class="fas fa-lock mr-2"></i> 
                            <?php echo getUserBalance($user_id) >= $total_amount ? 'Complete Purchase' : 'Insufficient Balance'; ?>
                        </button>
                        </form>
                        
                        <?php if(getUserBalance($user_id) < $total_amount): ?>
                            <div class="mt-4">
                                <a href="deposit.php" 
                                   class="block w-full text-center bg-gradient-to-r from-blue-600 to-indigo-600 text-white font-bold py-3 px-6 rounded-lg hover:from-blue-700 hover:to-indigo-700 transition duration-300">
                                    <i class="fas fa-plus-circle mr-2"></i> Deposit Now
                                </a>
                            </div>
                        <?php endif; ?>
                        
                        <!-- Payment Security -->
                        <div class="mt-6 pt-6 border-t">
                            <div class="flex items-center justify-center space-x-6 text-gray-600">
                                <div class="text-center">
                                    <i class="fas fa-shield-alt text-green-500 text-2xl mb-2"></i>
                                    <p class="text-xs">100% Secure</p>
                                </div>
                                <div class="text-center">
                                    <i class="fas fa-bolt text-yellow-500 text-2xl mb-2"></i>
                                    <p class="text-xs">Instant Delivery</p>
                                </div>
                                <div class="text-center">
                                    <i class="fas fa-headset text-blue-500 text-2xl mb-2"></i>
                                    <p class="text-xs">24/7 Support</p>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Support Card -->
                    <div class="bg-gradient-to-r from-blue-500 to-indigo-600 rounded-xl shadow-lg p-6 text-white animate__animated animate__fadeInRight" style="animation-delay: 0.1s">
                        <h3 class="text-xl font-bold mb-4">
                            <i class="fas fa-question-circle mr-2"></i> Need Help?
                        </h3>
                        <p class="mb-6 opacity-90">
                            Our support team is available 24/7 to assist you with checkout.
                        </p>
                        <div class="space-y-3">
                            <a href="https://t.me/support" target="_blank" 
                               class="flex items-center justify-center bg-white text-blue-600 font-bold py-3 px-4 rounded-lg hover:bg-blue-50 transition duration-300">
                                <i class="fab fa-telegram mr-2 text-xl"></i> Telegram Support
                            </a>
                            <a href="https://wa.me/8801234567890" target="_blank" 
                               class="flex items-center justify-center bg-white text-green-600 font-bold py-3 px-4 rounded-lg hover:bg-green-50 transition duration-300">
                                <i class="fab fa-whatsapp mr-2 text-xl"></i> WhatsApp Support
                            </a>
                        </div>
                    </div>
                    
                    <!-- Return to Cart -->
                    <div class="bg-white rounded-xl shadow-lg p-6 animate__animated animate__fadeInRight" style="animation-delay: 0.2s">
                        <a href="cart.php" 
                           class="flex items-center justify-center text-gray-700 hover:text-blue-600 font-medium transition duration-300">
                            <i class="fas fa-arrow-left mr-2"></i> Return to Cart
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script>
    // Form validation
    document.addEventListener('DOMContentLoaded', function() {
        const form = document.querySelector('form');
        const telegramInput = form.querySelector('[name="telegram_username"]');
        const whatsappInput = form.querySelector('[name="whatsapp_number"]');
        const submitBtn = form.querySelector('button[type="submit"]');
        
        function validateForm() {
            if(!telegramInput.value.trim() && !whatsappInput.value.trim()) {
                submitBtn.disabled = true;
                submitBtn.innerHTML = '<i class="fas fa-exclamation-circle mr-2"></i> Provide Contact Info';
                submitBtn.classList.remove('from-green-500', 'to-emerald-600');
                submitBtn.classList.add('from-red-500', 'to-red-600');
            } else {
                submitBtn.disabled = false;
                submitBtn.innerHTML = '<i class="fas fa-lock mr-2"></i> Complete Purchase';
                submitBtn.classList.remove('from-red-500', 'to-red-600');
                submitBtn.classList.add('from-green-500', 'to-emerald-600');
            }
        }
        
        telegramInput.addEventListener('input', validateForm);
        whatsappInput.addEventListener('input', validateForm);
        
        // Initial validation
        validateForm();
        
        // Add animations
        const steps = document.querySelectorAll('.checkout-step');
        steps.forEach((step, index) => {
            setTimeout(() => {
                step.classList.add('animate__animated', 'animate__fadeInUp');
            }, index * 200);
        });
    });
    
    // Show confirmation before submitting
    document.querySelector('form').addEventListener('submit', function(e) {
        if(!confirm('Are you sure you want to complete this purchase?')) {
            e.preventDefault();
        }
    });
    </script>
</body>
</html>